<?php

namespace App\Models;

use CodeIgniter\Model;

class BlogModel extends Model
{
    protected $db;
    protected $blog  = 'blog';
    protected $category  = 'blog_category';
    protected $blog_review  = 'blog_review';

    function get_all_blog_for_front()
    {

        return $this->db->table($this->blog)
            ->select('blog_category.category_name, blog.title, blog.image, blog.status, blog.id, blog.author, blog.date, blog.title_slug, blog.short_description')
            ->select('CASE 
                         WHEN blog.author = 1 THEN "Admin" 
                         ELSE users.user_name 
                     END AS author_name', false) // false prevents automatic escaping
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->join('users', 'users.user_id = blog.author', 'left')
            ->where('blog.status', 1)
            ->get()
            ->getResult();
    }

    function get_all_blog()
    {
        return $this->db->table($this->blog)
            ->select('blog_category.category_name, blog.title, blog.image, blog.status, blog.id, blog.author')
            ->select('CASE 
                         WHEN blog.author = 1 THEN "Admin" 
                         ELSE users.user_name 
                     END AS author_name', false) // false prevents automatic escaping
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->join('users', 'users.user_id = blog.author', 'left') // Left join to avoid issues if author=1
            ->get()
            ->getResult();
    }



    function get_all_blog_by_author($id)
    {
        return $this->db->table($this->blog)
            ->where('author', $id)
            ->select('blog_category.category_name, blog.title, blog.image, blog.status, blog.id')
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->get()
            ->getResult();
    }

    function get_all_blog_for_all_blog_page()
    {
        return $this->db->table($this->blog)
            ->select('title, title_slug, image, short_description, date, view')
            ->get()
            ->getResult();
    }

    function get_top_10_blog()
    {
        return $this->db->table($this->blog)
            ->select('*')
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->limit(10)
            ->orderBy('id', 'DESC')
            ->get()
            ->getResult();
    }

    function get_top_4_blog()
    {
        return $this->db->table($this->blog)
            ->select('title, title_slug, image, date')
            ->limit(4)
            ->orderBy('id', 'DESC')
            ->where('status', 1)
            ->get()
            ->getResult();
    }

    function get_blog_by_id($id)
    {
        return $this->db->table($this->blog)
            ->where(['id' => $id])
            ->get()
            ->getRow();
    }

    function get_blog_by_category($cat_id)
    {

        return $this->db->table($this->blog)
            ->select('blog_category.category_name, blog.title, blog.image, blog.status, blog.id, blog.author, blog.date, blog.title_slug, blog.short_description ')
            ->select('CASE 
                         WHEN blog.author = 1 THEN "Admin" 
                         ELSE users.user_name 
                     END AS author_name', false) // false prevents automatic escaping
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->join('users', 'users.user_id = blog.author', 'left')
            ->where('category', $cat_id)
            ->where('status', 1)
            ->get()
            ->getResult();
    }

    function get_blog_by_slug($slug)
    {
        return $this->db->table($this->blog)
            ->select('blog_category.*, blog.*')
            ->select('CASE 
                         WHEN blog.author = 1 THEN "Admin" 
                         ELSE users.user_name 
                     END AS author_name', false) // false prevents automatic escaping
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->join('users', 'users.user_id = blog.author', 'left')
            ->where('title_slug', $slug)
            ->where('status', 1)
            ->get()
            ->getRow();
    }

    function add_new_blog($data)
    {
        return $this->db->table($this->blog)->insert($data);
    }

    function update_blog_details($data, $id)
    {
        return $this->db->table($this->blog)
            ->where('id',  $id)
            ->update($data);
    }

    function delete_blog($id)
    {
        return $this->db->table($this->blog)
            ->where('id',  $id)
            ->delete();
    }

    function get_all_blog_by_category_slug($slug)
    {
        return $this->db->table($this->category)
            ->select('*')
            ->join('blog', 'blog.category = blog_category.category_id')
            ->where('category_slug', $slug)
            ->get()
            ->getResult();
    }

    function get_all_category()
    {
        return $this->db->table($this->category)->get()->getResult();
    }

    function get_category_by_id($id)
    {
        return $this->db->table($this->category)
            ->where(['category_id' => $id])
            ->get()
            ->getRow();
    }

    function get_category_by_slug($slug)
    {
        return $this->db->table($this->category)
            ->where(['category_slug' => $slug])
            ->get()
            ->getRow();
    }

    function add_new_category($data)
    {
        return $this->db->table($this->category)->insert($data);
    }

    function update_category($data, $id)
    {
        return $this->db->table($this->category)
            ->where(['category_id' => $id])
            ->update($data);
    }

    function delete_category($id)
    {
        return $this->db->table($this->category)
            ->where(['category_id' => $id])
            ->delete();
    }

    function submit_blog_review($data)
    {
        return $this->db->table($this->blog_review)->insert($data);
    }

    function get_all_blog_review()
    {
        return $this->db->table($this->blog_review)->get()->getResult();
    }

    function delete_blog_review($id)
    {
        return $this->db->table($this->blog_review)
            ->where(['id' => $id])
            ->delete();
    }

    function get_all_blog_four_for_frontend()
    {
        return $this->db->table($this->blog)
            ->select('blog_category.category_name, blog.title, blog.image, blog.status, blog.id, blog.author, blog.date, blog.title_slug ')
            ->select('CASE 
                         WHEN blog.author = 1 THEN "Admin" 
                         ELSE users.user_name 
                     END AS author_name', false) // false prevents automatic escaping
            ->join('blog_category', 'blog_category.category_id = blog.category')
            ->join('users', 'users.user_id = blog.author', 'left')
            ->where('blog.status', 1)
            ->limit(4)
            ->get()
            ->getResult();
    }

    function get_top_5_blog()
    {
        return $this->db->table($this->blog)
            ->select('title, title_slug, date, image')
            ->where('status', 1)
            ->limit(5)
            ->orderBy('id', "desc")
            ->get()
            ->getResult();
    }

    function get_all_blog_for_frontend()
    {
        return $this->db->table($this->blog)
            ->select('title, title_slug, short_description, date, view, image')
            ->where('status', 1)
            ->get()
            ->getResult();
    }

    function get_all_blog_by_category_slug_for_frontend($slug)
    {
        $cat_id = $this->get_category_by_slug($slug);
        return $this->db->table($this->blog)
            ->select('title, title_slug, short_description, date, view, image')
            ->where('category', $cat_id->category_id)
            ->where('status', 1)
            ->get()
            ->getResult();
    }

    function get_single_blog_for_frontend()
    {
        return $this->db->table($this->blog)
            ->select('title, title_slug, short_description, image')
            ->orderBy('id', 'RANDOM')
            ->where('status', 1)
            ->limit(1)
            ->get()

            ->getRow();
    }
}
